#include "ibm.h"

#include "mmsystem.h"

#include <stdio.h>
#include <math.h>

#include "aa.h"

#define NOISELENGTH (128 * KILOBYTE)
#define DURATION                2000 // in milliseconds
#define TONELENGTH  ((int) (((double) 11025.0 * DURATION) / 1000.0 + 0.5))

// #define DEBUGSOUND

IMPORT UBYTE          memory[32768];
IMPORT FLAG           consoleopen,
                      psgplaying[14];
IMPORT int            machine;

MODULE FLAG           PSGOpened[14]   = {FALSE, FALSE, FALSE, FALSE, FALSE, FALSE,
                                         FALSE, FALSE, FALSE, FALSE, FALSE, FALSE,
                                         FALSE, FALSE},
                      PSGPrepared[14] = {FALSE, FALSE, FALSE, FALSE, FALSE, FALSE,
                                         FALSE, FALSE, FALSE, FALSE, FALSE, FALSE,
                                         FALSE, FALSE};
MODULE HWAVEOUT       hPSG[14];
MODULE WAVEHDR        PSGHdr[14];
MODULE UBYTE          ToneBuffer[7][TONELENGTH],
                      NoiseBuffer[7][NOISELENGTH];

MODULE void generatewave(int channel, double hertz, int volume);

EXPORT void OpenChannels(void)
{   WAVEFORMATEX l_WaveFormatEx;
    int          i;

    for (i = 0; i < 14; i++)
    {   l_WaveFormatEx.wFormatTag      = WAVE_FORMAT_PCM;
        l_WaveFormatEx.nChannels       = 1;
        l_WaveFormatEx.nSamplesPerSec  = 11025;
        l_WaveFormatEx.wBitsPerSample  = 8;
        l_WaveFormatEx.nAvgBytesPerSec = 11025;
        l_WaveFormatEx.nBlockAlign     = 1;
     // l_WaveFormatEx.cbSize          = 0; ignored
        memset(&hPSG[i], 0, sizeof(HWAVEOUT));
        DISCARD waveOutOpen
        (   &hPSG[i],
            WAVE_MAPPER,
            &l_WaveFormatEx,
            0,
            0,
            CALLBACK_NULL
        );
        PSGOpened[i] = TRUE;
}   }

EXPORT void TonePlay(int channel, double hertz, int volume)
{   // assert(volume >= 1 && volume <= 15);

#ifdef DEBUGSOUND
    if (psgplaying[channel] || PSGPrepared[channel])
    {   OPENCONSOLE; printf("Channel %ld is already playing a tone!\n", channel); REACTIVATE;
        return;
    } else
    {   OPENCONSOLE; printf("Playing %lf Hz tone on channel %ld!\n", hertz, channel); REACTIVATE;
    }
#endif

    // free old buffer (if required)
    // PSGStop(channel); done in ibm.c
    generatewave(channel, hertz, volume);

    PSGHdr[channel].lpData           = ToneBuffer[channel];
    PSGHdr[channel].dwBufferLength   = TONELENGTH;
    PSGHdr[channel].dwBytesRecorded  = 0;
    PSGHdr[channel].dwUser           = 0;
    PSGHdr[channel].dwFlags          = 0;
    PSGHdr[channel].dwLoops          = ~0; // loop almost indefinitely
    PSGHdr[channel].lpNext           = NULL;
    PSGHdr[channel].reserved         = (unsigned long) NULL;
    DISCARD waveOutPrepareHeader(hPSG[channel], &PSGHdr[channel], sizeof(WAVEHDR));
    PSGPrepared[channel] = TRUE;
    PSGHdr[channel].dwFlags          |= WHDR_BEGINLOOP | WHDR_ENDLOOP;
    DISCARD waveOutWrite(hPSG[channel], &PSGHdr[channel], sizeof(WAVEHDR));
}

EXPORT void NoisePlay(int channel, int volume)
{   AUTO    ULONG i;
    PERSIST UBYTE a_start[8]   =
    { 128, 110,  91,  73,  55,  37,  18,   0};
    PERSIST UBYTE a_scaling[8] =
    {   0,  36,  73, 109, 145, 182, 219, 255};
    PERSIST UBYTE ie_start[16]   =
    { 128, 120, 110, 104,  96,  88,  80,  72,
       64,  48,  40,  32,  24,  16,   8,   0};
    PERSIST UBYTE ie_scaling[16] =
    {   0,  17,  34,  51,  68,  85, 102, 119,
      136, 153, 170, 187, 204, 221, 238, 255};

    /* Arcadia:
        0: 128 (128      ) + 0..  0       (  0) = 128 average (128 + (  0/2) =       128)
        1: 110 (109.7143 ) + 0.. 36.42857 ( 36) = 128 average (110 + ( 36/2) = about 128)
        2:  91 ( 91.42857) + 0.. 72.85714 ( 73) = 128 average ( 91 + ( 73/2) = about 128)
        3:  73 ( 73.14286) + 0..109.2857  (109) = 128 average ( 73 + (109/2) = about 128)
        4:  55 ( 54.85714) + 0..145.7143  (146) = 128 average ( 55 + (146/2) = about 128)
        5:  37 ( 36.57143) + 0..182.1429  (182) = 128 average ( 37 + (182/2) = about 128)
        6:  18 ( 18.28571) + 0..218.5714  (219) = 128 average ( 18 + (219/2) = about 128)
        7:   0 (  0      ) + 0..255       (255) = 128 average (  0 + (255/2) = about 128)

    Interton/Elektor:
        0: 128 + 0..  0 = 128 average
        1: 120 + 0.. 17 = 128 average
        2: 112 + 0.. 34 = 128 average
        3: 104 + 0.. 51 = 128 average
        4:  96 + 0.. 68 = 128 average
        5:  88 + 0.. 85 = 128 average
        6:  80 + 0..102 = 128 average
        7:  72 + 0..119 = 128 average
        8:  64 + 0..136 = 128 average
        9:  48 + 0..153 = 128 average
       10:  40 + 0..170 = 128 average
       11:  32 + 0..187 = 128 average
       12:  24 + 0..204 = 128 average
       13:  16 + 0..221 = 128 average
       14:   8 + 0..238 = 128 average
       15:   0 + 0..255 = 128 average

    Strictly speaking, the ideal average is maybe 127.5, not 128? */

    // assert(volume >= 1 && volume <= 15);

#ifdef DEBUGSOUND
    if (psgplaying[channel] || PSGPrepared[channel])
    {   OPENCONSOLE; printf("Channel %ld is already playing a noise!\n", channel); REACTIVATE;
        return;
    } else
    {   OPENCONSOLE; printf("Playing noise on channel %ld!\n", channel); REACTIVATE;
    }
#endif

    // free old buffer (if required)
    // PSGStop(channel); done in ibm.c
    if (machine == ARCADIA)
    {   for (i = 0; i < NOISELENGTH; i++)
        {   NoiseBuffer[channel - 7][i] = a_start[volume] + (rand() % (a_scaling[volume]));
    }   }
    else
    {   // assert(machine == INTERTON || machine == ELEKTOR);
        for (i = 0; i < NOISELENGTH; i++)
        {   NoiseBuffer[channel - 7][i] = ie_start[volume] + (rand() % (ie_scaling[volume]));
    }   }

    PSGHdr[channel].lpData          = NoiseBuffer[channel - 7];
    PSGHdr[channel].dwBufferLength  = NOISELENGTH;
    PSGHdr[channel].dwBytesRecorded = 0;
    PSGHdr[channel].dwUser          = 0;
    PSGHdr[channel].dwFlags         = 0;
    PSGHdr[channel].dwLoops         = ~0; // loop almost indefinitely
    PSGHdr[channel].lpNext          = NULL;
    PSGHdr[channel].reserved        = (unsigned long) NULL;
    DISCARD waveOutPrepareHeader(hPSG[channel], &PSGHdr[channel], sizeof(WAVEHDR));
    PSGPrepared[channel] = TRUE;
    PSGHdr[channel].dwFlags         |= WHDR_BEGINLOOP | WHDR_ENDLOOP;
    DISCARD waveOutWrite(hPSG[channel], &PSGHdr[channel], sizeof(WAVEHDR));
}

EXPORT void PSGStop(int channel)
{
#ifdef DEBUGSOUND
    OPENCONSOLE; printf("Stopping channel %ld...\n", channel);
#endif
    if (psgplaying[channel]) // if (PSGPrepared[channel])
    {   psgplaying[channel] = FALSE;
        DISCARD waveOutReset(hPSG[channel]);
        DISCARD waveOutUnprepareHeader(hPSG[channel], &PSGHdr[channel], sizeof(WAVEHDR));
        PSGPrepared[channel] = FALSE;
#ifdef DEBUGSOUND
        printf("OK, stopped channel %ld.\n", channel); REACTIVATE;
#endif
    }
#ifdef DEBUGSOUND
    else printf("Didn't need to stop channel %ld.\n", channel); REACTIVATE;
#endif
}

EXPORT void PSGClose(int channel)
{   if (PSGOpened[channel])
    {   PSGStop(channel);
        DISCARD waveOutClose(hPSG[channel]);
        PSGOpened[channel] = FALSE;
}   }

MODULE void generatewave(int channel, double hertz, int volume)
{   AUTO    double buf_perc;
    AUTO    double perc;
    AUTO    double swap;
    AUTO    double split;
    AUTO    double rate_ratio = 1000.0 / DURATION;
    AUTO    int    buf_size   = 100;
    AUTO    int    i,
                   j,
                   local_limit;
    AUTO    int    output_val;
    PERSIST UBYTE a_scaling[8] =
    {   0, //  0: 128 +/-   0
       18, //  1: 128 +/-  18 ( 18.14286)
       36, //  2: 128 +/-  32 ( 36.28571)
       54, //  3: 128 +/-  54 ( 54.42857)
       73, //  4: 128 +/-  73 ( 72.57143)
       91, //  5: 128 +/-  91 ( 90.71429)
      109, //  6: 128 +/- 109 (108.8571 )
      127  //  7: 128 +/- 127
    };
    PERSIST UBYTE ie_scaling[16] =
    {   0, //  0: 128 +/-   0
        8, //  1: 128 +/-   8 (  8.46667)
       17, //  2: 128 +/-  17 ( 16.93333)
       25, //  3: 128 +/-  25 ( 25.4    )
       34, //  4: 128 +/-  34 ( 33.86667)
       42, //  5: 128 +/-  42 ( 42.33333)
       51, //  6: 128 +/-  51 ( 50.8    )
       59, //  7: 128 +/-  59 ( 59.26667)
       68, //  8: 128 +/-  68 ( 67.73333)
       76, //  9: 128 +/-  76 ( 76.2    )
       85, // 10: 128 +/-  85 ( 84.66667)
       93, // 11: 128 +/-  93 ( 93.13333)
      102, // 12: 128 +/- 102 (101.6    )
      110, // 13: 128 +/- 110 (110.06667)
      119, // 14: 128 +/- 119 (118.53333)
      127  // 15: 128 +/- 127
    };

#ifdef DEBUGSOUND
    OPENCONSOLE; printf("Tone volume for channel %ld is %ld!\n", channel, volume); REACTIVATE;
#endif

    buf_perc = (double) buf_size / (double) TONELENGTH;
    swap = (1.0 / (2.0 * hertz) * rate_ratio);

    if (machine == ARCADIA)
    {   // is this function harmless even if volume is 0?
        // assert(volume >= 1 && volume <= 7);

        for (i = 0; i < TONELENGTH; i += buf_size)
        {   // for entire period

            if (i + buf_size > TONELENGTH)
            {   local_limit = TONELENGTH - i;
            } else
            {   local_limit = buf_size;
            }

            for (j = 0; j < local_limit; j++)
            {   // fill up local buffer

                // determine where in sample we are
                perc = (double)(i + j) / (double)(TONELENGTH);

                // from where in sample we are can now determine what we should be
                split = perc / swap;
                output_val = ((int) (split + 0.5) % 2) ? (128 + a_scaling[volume]) : (128 - a_scaling[volume]);
                // strictly speaking, it should be relevant to 127.5 not 128?
                ToneBuffer[channel][i + j] = (unsigned char)output_val;
    }   }   }
    else
    {   // assert(machine == INTERTON || machine == ELEKTOR)
        // is this function harmless even if volume is 0?
        // assert(volume >= 1 && volume <= 15);

        for (i = 0; i < TONELENGTH; i += buf_size)
        {   // for entire period

            if (i + buf_size > TONELENGTH)
            {   local_limit = TONELENGTH - i;
            } else
            {   local_limit = buf_size;
            }

            for (j = 0; j < local_limit; j++)
            {   // fill up local buffer

                // determine where in sample we are
                perc = (double)(i + j) / (double)(TONELENGTH);

                // from where in sample we are can now determine what we should be
                split = perc / swap;
                output_val = ((int) (split + 0.5) % 2) ? (128 + ie_scaling[volume]) : (128 - ie_scaling[volume]);
                // strictly speaking, it should be relevant to 127.5 not 128?
                ToneBuffer[channel][i + j] = (unsigned char)output_val;
}   }   }   }
